/**
 * \file: exchnd_names.c
 *
 * Strings describing the exception handlers are implemented here.
 *
 * \component: exchndd
 *
 * \author: Matthias Weise (mweise@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <linux/exchnd.h>

#include "exchnd_names.h"

/* Define the strings of the exception handler */
const char *exchnd_trig_names[] = EXCHND_TRIGGER_STRINGS;
const char *exchnd_mod_names[] = EXCHND_MODULE_STRINGS;
const char *exchnd_sig_names[] = EXCHND_SIGNAL_STRINGS;

/*
 * CONF strings
 * Note: Should be aligned with linux/exchnd.h
 */
const char *exchnd_conf_names[EC_LAST_ELEMENT + 1] = {
    "NONE",
    "EC_NO_MODULES",
    "EC_SAFETY",
    "EC_DEBUG",
    "EC_VERBOSE",
#ifdef EC_RBCM
    "EC_RBCM",
#endif
#ifdef EC_DN_NORESTART
    "EC_DN_NORESTART",
#endif
#ifdef EC_DN
    "EC_DN",
#endif
    "EC_LAST_ELEMENT"
};

/*
 * SIGNAL CONF strings
 * Note: Should be aligned with linux/exchnd.h
 */
const char *exchnd_signal_conf_names[ESC_LAST_ELEMENT + 1] = {
    "NONE",
    "ESC_NO_MODULES",
    "ESC_SAFETY",
    "ESC_DEBUG",
    "ESC_VERBOSE",
    "ESC_ALL_MODULES",
#ifdef ESC_RBCM_NORESTART
    "ESC_RBCM_NORESTART",
#endif
#ifdef ESC_RBCM_RESTART
    "ESC_RBCM_RESTART",
#endif
#ifdef ESC_DN_NORESTART
    "ESC_DN_NORESTART",
#endif
#ifdef ESC_DN_RESTART
    "ESC_DN_RESTART",
#endif
    "ESC_LAST_ELEMENT"
};


/*
 * \func conf_name
 *
 * Converts conf enum to conf name.
 * The function returns the name of the given conf. If an invalid conf
 * value is given the function returns the none conf name
 *
 * \param conf Trigger enum value to covnert
 *
 * \return String describing the conf
 */
const char *conf_name(enum exchnd_conf conf)
{
    if (conf < ARRAY_SIZE(exchnd_conf_names))
        return exchnd_conf_names[conf];
    else
        /* Return empty conf ET_NONE */
        return exchnd_conf_names[0];
}

/*
 * \func signal_conf_name
 *
 * Converts conf enum to signal_conf name.
 * The function returns the name of the given signal_conf. If an invalid conf
 * value is given the function returns the none signal_conf name
 *
 * \param signal_conf Trigger enum value to convert
 *
 * \return String describing the signal_conf
 */
const char *signal_conf_name(enum exchnd_signal_conf conf)
{
    if (conf < ARRAY_SIZE(exchnd_signal_conf_names))
        return exchnd_signal_conf_names[conf];
    else
        /* Return empty signal_conf ET_NONE */
        return exchnd_signal_conf_names[0];
}

/*
 * \func trigger_name
 *
 * Converts trigger enum to trigger name.
 * The function returns the name of the given trigger. If an invalid trigger
 * value is given the function returns the none trigger name
 *
 * \param trigger Trigger enum value to convert
 *
 * \return String describing the trigger
 */
const char *trigger_name(enum exchnd_triggers trigger)
{
    if (trigger < ARRAY_SIZE(exchnd_trig_names))
        return exchnd_trig_names[trigger];
    else
        /* Return empty trigger ET_NONE */
        return exchnd_trig_names[0];
}

/*
 * \func module_name
 *
 * Converts module enum to module name.
 * The function returns the name of the given module. If an invalid module
 * value is given the function returns the none module name
 *
 * \param module Module enum value to convert
 *
 * \return String describing the module
 */
const char *module_name(enum exchnd_modules module)
{
    if (module < ARRAY_SIZE(exchnd_mod_names))
        return exchnd_mod_names[module];
    else
        /* Return empty module EHM_NONE */
        return exchnd_mod_names[0];
}

/*
 * \func signal_name
 *
 * Converts signal number to module name.
 * The function returns the name of the given signal number. If an invalid
 * signal number  is given the function returns the none signal name
 *
 * \param module Module enum value to convert
 *
 * \return String describing the module
 */
const char *signal_name(int sig)
{
    if ((unsigned int)sig < ARRAY_SIZE(exchnd_sig_names))
        return exchnd_sig_names[sig];
    else
        /* Return empty signal NONE */
        return exchnd_sig_names[0];
}

/*
 * \func get_trigger
 *
 * Converts trigger name to trigger enum
 * The function returns the enum of the given trigger string. If an invalid
 * trigger string is given the function returns the last element marker of the
 * trigger enum
 *
 * \param trig_name String describing the trigger
 *
 * \return Enum for the trigger described by the given string
 */
enum exchnd_triggers get_trigger(const char *trig_name)
{
    enum exchnd_triggers i;

    for (i = ET_NONE; i < ET_LAST_ELEMENT; i++) {
        if (strlen(exchnd_trig_names[i]) != strlen(trig_name))
            continue;

        if (!strncmp(trig_name,
                     exchnd_trig_names[i],
                     strlen(exchnd_trig_names[i])))
            return i;
    }

    return ET_LAST_ELEMENT;
}


/*
 * \func get_module
 *
 * Converts module name to module enum
 * The function returns the enum of the given module string. If an invalid
 * module string is given the function returns the last element marker of the
 * module enum
 *
 * \param mod_name String describing the module
 *
 * \return Enum for the module described by the given string
 */
enum exchnd_modules get_module(const char *mod_name)
{
    enum exchnd_modules i;

    for (i = EHM_NONE; i < EHM_LAST_ELEMENT; i++) {
        if (strlen(exchnd_mod_names[i]) != strlen(mod_name))
            continue;

        if (!strncmp(mod_name,
                     exchnd_mod_names[i],
                     strlen(exchnd_mod_names[i])))
            return i;
    }

    return EHM_LAST_ELEMENT;
}

/*
 * \func get_signal
 *
 * Converts signal name to signal number
 * The function returns the number of the given signal string. If an invalid
 * signal string is given the function returns  0
 * trigger enum
 *
 * \param trigger_name String describing the trigger
 *
 * \return Enum for the trigger described by the given string
 */
int get_signal(const char *sig_name)
{
    unsigned int i;

    if (!strncmp(sig_name, "SIGIOT", strlen("SIGIOT")) ||
        !strncmp(sig_name, "SIGABRT", strlen("SIGABRT")))
        return SIGIOT;

    if (!strncmp(sig_name, "SIGIO", strlen("SIGIO")) ||
        !strncmp(sig_name, "SIGPOLL", strlen("SIGPOLL")))
        return SIGIO;

    for (i = 0; i < ARRAY_SIZE(exchnd_sig_names); i++) {
        if (strlen(exchnd_sig_names[i]) != strlen(sig_name))
            continue;

        if (!strncmp(sig_name,
                     exchnd_sig_names[i],
                     strlen(exchnd_sig_names[i])))
            return i;
    }

    return 0;
}

/*
 * \func get_conf
 *
 * Converts conf name to conf enum
 * The function returns the enum of the given conf string. If an invalid
 * conf string is given the function returns the last element marker of the
 * conf enum
 *
 * \param config_name String describing the configuration
 *
 * \return Enum for the conf described by the given string
 */
enum exchnd_conf get_conf(const char *config_name)
{
    enum exchnd_conf i;

    for (i = EC_NO_MODULES; i < EC_LAST_ELEMENT; i++) {
        if (strlen(exchnd_conf_names[i]) != strlen(config_name))
            continue;

        if (!strncmp(config_name,
                     exchnd_conf_names[i],
                     strlen(exchnd_conf_names[i])))
            return i;
    }

    return EC_LAST_ELEMENT;
}

/*
 * \func get_sig_conf
 *
 * Converts conf name to conf enum
 * The function returns the enum of the given conf string. If an invalid
 * conf string is given the function returns the last element marker of the
 * conf enum
 *
 * \param config_name String describing the configuration
 *
 * \return Enum for the conf described by the given string
 */
enum exchnd_signal_conf get_sig_conf(const char *config_name)
{
    enum exchnd_signal_conf i;

    for (i = ESC_NO_MODULES; i < ESC_LAST_ELEMENT; i++) {
        if (strlen(exchnd_signal_conf_names[i]) != strlen(config_name))
            continue;

        if (!strncmp(config_name,
                     exchnd_signal_conf_names[i],
                     strlen(exchnd_signal_conf_names[i])))
            return i;
    }

    return ESC_LAST_ELEMENT;
}